﻿using System;
using System.Globalization;
using System.IO.Ports;
using System.Threading;
using System.Windows.Forms;
using System.Windows.Forms.DataVisualization.Charting;

namespace TestRealtimeChart2 {
    public partial class Form1 : Form {

        // Array that holds the values.
        internal static double[] values;
        // To keep track of where the latest value was written in the array.
        internal static int currentValueIndex;
        // How many values to keep before starting from the beginning.
        internal const int MAX_VALUE_COUNT = 1000;

        public Form1() {
            InitializeComponent();

            // Set the correct culture so a dot is parsed as a decimal point.
            CultureInfo customCulture = (CultureInfo) Thread.CurrentThread.CurrentCulture.Clone();
            customCulture.NumberFormat.NumberDecimalSeparator = ".";
            Thread.CurrentThread.CurrentCulture = customCulture;
            
            // Start a new thread so the UI does not freeze.
            Thread thread1 = new Thread(ReadValues);
            thread1.Start();

            // Initialise the array with the necessary amount of spaces.
            values = new double[MAX_VALUE_COUNT];

            // Make the Y-axis pretty.
            chart1.ChartAreas[0].AxisX.Minimum = 0;
            chart1.ChartAreas[0].AxisX.Maximum = MAX_VALUE_COUNT;
        }

        private void ReadValues() {

            // Initialize the COM port.
            SerialPort serialPort = new SerialPort("COM5");
            serialPort.Open();

            while (true) {

                // Read a value from serial.
                // A Try is used because serial provides gibberish at the start.
                double.TryParse(serialPort.ReadLine(), out double currentValue);

                // Save the value at the correct spot in the array.
                values[currentValueIndex] = currentValue;

                // Set the index to the next spot.
                currentValueIndex++;

                // If at the end, start from the beginning.
                if (currentValueIndex >= MAX_VALUE_COUNT) {
                    currentValueIndex = 0;
                }
            }
        }

        private void ChartRefreshTimer_Tick(object sender, EventArgs e) {

            // Empty the chart.
            chart1.Series[0].Points.Clear();
            // Stop the UI from updating the chart for a moment.
            chart1.Series[0].Points.SuspendUpdates();

            // Add the entire array as points.
            for (int i = 0; i < values.Length; i++) {
                if (i == currentValueIndex) {
                    // This point marks the newest read value and is styled differently.
                    DataPoint point = new DataPoint(i, values[i]);
                    point.MarkerStyle = MarkerStyle.Circle;
                    point.MarkerSize = 10;
                    chart1.Series[0].Points.Add(point);
                } else {
                    chart1.Series[0].Points.AddXY(i, values[i]);
                }
            }

            // Allow the UI to update the chart again.
            chart1.Series[0].Points.ResumeUpdates();
        }
    }
}
